const RSAKeyError = require('./error');

/**
 * Extracts 32-bit integer number from a partial buffer
 * @private
 * @param {buffer} buffer
 * @param {number} [offset=0]
 * @return {number}
 */
function get32IntFromBuffer(buffer, offset = 0) {
  let size = buffer.length - offset;
  if (size < 0) { return NaN; }
  else if (size >= 4) { return buffer.readUInt32BE(offset); }
  else {
    return [...buffer.slice(offset)].reduce((acc, val, i, arr) => {
      return acc + val * Math.pow(16, (arr.length - i - 1) * 2);
    }, 0);
  }
}

/**
 * Checks if all components for a public key are present
 * @private
 * @param {object} components
 * @return {boolean}
 */
function hasPublicComponents(components) {
  return typeof components.n !== 'undefined' && typeof components.e !== 'undefined';
}

/**
 * Checks if all components for a private key are present
 * @private
 * @param {object} components
 * @return {boolean}
 */
function hasPrivateComponents(components) {
  const required = ['n', 'e', 'd', 'p', 'q', 'dp', 'dq', 'qi'];
  for (let name of required) {
    if (typeof components[name] === 'undefined') { return false; }
  }
  return true;
}

/**
 * Ensures string output, converting from buffer with specified encoding if necessary
 * @param {string|buffer} input
 * @param encoding
 * @return {string}
 */
function bufferStringify(input, encoding) {
  if (typeof input === 'string') {
    return input;
  }
  else if (input instanceof Buffer) {
    return input.toString(encoding);
  }
  else {
    throw new RSAKeyError('Buffer or string is required');
  }
}

module.exports = { get32IntFromBuffer, hasPrivateComponents, hasPublicComponents, bufferStringify };