const pem = require('./formats/pem');
const der = require('./formats/der');
const { PEM, DER } = require('./constants');
const RSAKeyError = require('./error');

/**
 * Decodes RSA key provided in any supported format into components
 * @param {string|buffer} key
 * @return {object}
 */
const decode = key => {

  /* Check for PEM format */
  let info = pem.testPEM(key);
  if (info !== null) {
    return pem.decodePEM(key, info);
  }

  /* Check for KER format */
  if (der.isDER(key)) {
    return der.decodeDER(key);
  }

  throw new RSAKeyError('Could not recognize key format');

};

/**
 * Encodes components into a RSA key in specified format
 * @param {object} components
 * @param {string} format
 * @param {string} type
 * @param {string} syntax
 * @return {string|buffer}
 */
const encode = (components, format, type, syntax) => {
  switch (format) {
    case PEM:
      return pem.encodePEM(components, type, syntax);
    case DER:
      return der.encodeDER(components, type, syntax);
    default:
      throw new RSAKeyError(`Unknown format: ${format}`);
  }
};

module.exports = { decode, encode };