const { decode, encode } = require('./key');
const { hasPrivateComponents, hasPublicComponents } = require('./util');
const { BASE64, PUBLIC, PRIVATE, DER, PEM, PKCS8, PKCS1 } = require('./constants');
const RSAKeyError = require('./error');
const crypto = require('crypto');

class RSAKey {

  /**
   * @param {string|buffer} [key]
   */
  constructor(key) {
    this.components = {};
    this.cached = {};
    if (typeof key !== 'undefined') {
      this.importKey(key);
    }
  }

  /**
   * Checks if object has key data
   * @return {boolean}
   */
  hasKey() {
    return hasPublicComponents(this.components) || hasPrivateComponents(this.components);
  }

  /**
   * Checks if object is empty (has no key data)
   * @return {boolean}
   */
  isEmpty() {
    return !this.hasKey();
  }

  /**
   * Checks if object has private key
   * @return {boolean}
   */
  isPrivate() {
    return hasPrivateComponents(this.components);
  }

  /**
   * Checks if object has public key
   * @param {boolean} [strict=false]
   * @return {boolean}
   */
  isPublic(strict = false) {
    return strict
      ? hasPublicComponents(this.components) && hasPrivateComponents(this.components) === false
      : hasPublicComponents(this.components);
  }

  /**
   * Checks key type
   * @return {string|null} public|private
   */
  getType() {
    if (this.isPrivate() === true) { return PRIVATE; }
    else if (this.isPublic(true) === true) { return PUBLIC; }
    else { return null; }
  }

  /**
   * Returns key size in bits
   * @return {number}
   */
  getKeySize() {
    return this.hasKey()
      ? (Buffer.from(this.components.n, BASE64).length - 1) * 8
      : 0;
  }

  /**
   * Imports key
   * @param {string|buffer|RSAKey} key
   */
  importKey(key) {
    this.components = key instanceof RSAKey || (

      /* istanbul ignore next: Fallback for webpack-transpiled version */
      typeof key === 'object'
      && typeof key.constructor === 'function'
      && key.constructor.name === 'RSAKey'
      && typeof key.components === 'object'
      /* end of fallback */

    ) ? key.components : decode(key);
    this.cached = {};
  }

  /**
   * Throws if no key data exists
   * @param {string} [action]
   * @param {boolean} [requirePrivate=false]
   * @private
   */
  _requireKey(/* istanbul ignore next */ action = 'perform this action', requirePrivate = false) {
    if (this.hasKey() === false) {
      throw new RSAKeyError(`Cannot ${action}, because the key is empty`);
    }
    if (requirePrivate === true && this.isPrivate() === false) {
      throw new RSAKeyError(`Cannot ${action}, private key is required, this is a public key`);
    }
  }

  _cached(cacheKey, callback) {
    if (typeof this.cached[cacheKey] === 'undefined') {
      this.cached[cacheKey] = callback();
    }
    return this.cached[cacheKey];
  }

  /**
   * Exports key in a specified format
   * @param {...string} [specs]
   * @return {string|buffer}
   */
  exportKey(...specs) {
    this._requireKey('export');
    let type = this.getType();
    let format = PEM;
    let syntax = PKCS8;
    for (let param of specs) {
      if ([PRIVATE, PUBLIC].includes(param)) { type = param; }
      else if ([PEM, DER].includes(param)) { format = param; }
      else if ([PKCS1, PKCS8].includes(param)) { syntax = param; }
      else {
        throw new RSAKeyError(`Unknown key export parameter ${param}`);
      }
    }
    return this._cached(`exportKey ${format} ${type} ${syntax}`, () => {
      return encode(this.components, format, type, syntax);
    });
  }

  /**
   * Returns public key fingerprint
   * @return {string}
   */
  getFingerprint() {
    this._requireKey('create fingerprint');
    return this._cached('getFingerprint', () => {
      return crypto.createHash('md5').update(this.exportKey(PUBLIC, PKCS8, DER)).digest('hex').match(/.{2}/g).join(':');
    });
  }

}

module.exports = RSAKey;